<?php
/**
 * Name:    EL F2+ Auth
 * Author:  alexjob.travail@gmail
 *           alexjob.travail@gmail
 *           @alexandre
 *
 * Added Awesomeness: Alexandre Boris
 *
 * Created:  20.02.2024
 *
 * Requirements: PHP5.6 or above
 *
 * @package    EL F2+ Auth
 * @author     Alexandre Boris
 * @filesource
 */
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Class El_f2plus_auth
 */
class El_f2plus_auth
{
	/**
	 * account status ('not_activated', etc ...)
	 *
	 * @var string
	 **/
	protected $status;

	/**
	 * extra where
	 *
	 * @var array
	 **/
	public $_extra_where = array();

	/**
	 * extra set
	 *
	 * @var array
	 **/
	public $_extra_set = array();

	/**
	 * caching of users and their groups
	 *
	 * @var array
	 **/
	public $_cache_user_in_group;

	/**
	 * __construct
	 *
	 * @author Ben
	 */
	public function __construct()
	{
		$this->config->load('el_f2plus_auth', TRUE);
		$this->load->library(array('email'));
		$this->lang->load('el_f2plus_auth');
		$this->load->helper(array('cookie', 'language','url'));

		$this->load->library('session');

		$this->load->model('membres_model');

		$this->_cache_user_in_group =& $this->membres_model->_cache_user_in_group;
	
		$email_config = $this->config->item('email_config', 'el_f2plus_auth');

		if ($this->config->item('use_ci_email', 'el_f2plus_auth') && isset($email_config) && is_array($email_config))
		{
			$this->email->initialize($email_config);
		}

		$this->membres_model->trigger_events('library_constructor');
	}

	/**
	 * __call
	 *
	 * Acts as a simple way to call model methods without loads of stupid alias'
	 *
	 * @param $method
	 * @param $arguments
	 * @return mixed
	 * @throws Exception
	 */
	public function __call($method, $arguments)
	{
		if (!method_exists( $this->membres_model, $method) )
		{
			throw new Exception('Undefined method el_f2plus_auth::' . $method . '() called');
		}
		if($method == 'create_user')
		{
			return call_user_func_array(array($this, 'register'), $arguments);
		}
		if($method=='update_user')
		{
			return call_user_func_array(array($this, 'update'), $arguments);
		}
		return call_user_func_array( array($this->membres_model, $method), $arguments);
	}

	/**
	 * __get
	 *
	 * Enables the use of CI super-global without having to define an extra variable.
	 *
	 * I can't remember where I first saw this, so thank you if you are the original author. -Militis
	 *
	 * @access	public
	 * @param	$var
	 * @return	mixed
	 */
	public function __get($var)
	{
		return get_instance()->$var;
	}


	/**
	 * forgotten password feature
	 *
	 * @param $identity
	 * @return mixed boolean / array
	 * @author Mathew
	 */
	public function forgotten_password($idMembre,$identity)    //changed $email to $identity
	{
		if ( $this->membres_model->forgotten_password($idMembre,$identity) )   //changed
		{
			// Get user information
      		$identifier = $this->membres_model->identity_column; // use model identity column, so it can be overridden in a controller
      		//die('identifier : '.$identifier.' identity : '.$identity);  // changed to get_user_by_identity from email      		
      		$user = $this->where($identifier, $identity)->users()->row();
			if ($user)
			{
				return $user->forgotten_password_code;
				
			}
			else
			{
				return FALSE;
			}
		}
		else
		{
			return FALSE;
		}
	}

	/**
	 * forgotten_password_complete
	 *
	 * @param $code
	 * @author Mathew
	 * @return bool
	 */
	public function forgotten_password_complete($code)
	{
		$this->membres_model->trigger_events('pre_password_change');

		$identity = $this->config->item('identity', 'el_f2plus_auth');
		$profile  = $this->where('forgotten_password_code', $code)->users()->row(); //pass the code to profile

		if (!$profile)
		{
			$this->membres_model->trigger_events(array('post_password_change', 'password_change_unsuccessful'));
			$this->set_error('password_change_unsuccessful');
			return FALSE;
		}

		$new_password = $this->membres_model->forgotten_password_complete($code, $profile->salt);

		if ($new_password)
		{
			$data = array(
				'identity'     => $profile->{$identity},
				'new_password' => $new_password
			);
			if(!$this->config->item('use_ci_email', 'el_f2plus_auth'))
			{
				$this->set_message('password_change_successful');
				$this->membres_model->trigger_events(array('post_password_change', 'password_change_successful'));
					return $data;
			}
			else
			{
				$message = $this->load->view($this->config->item('email_templates', 'el_f2plus_auth').$this->config->item('email_forgot_password_complete', 'el_f2plus_auth'), $data, true);

				$this->email->clear();
				$this->email->from($this->config->item('admin_email', 'el_f2plus_auth'), $this->config->item('site_title', 'el_f2plus_auth'));
				$this->email->to($profile->email);
				$this->email->subject($this->config->item('site_title', 'el_f2plus_auth') . ' - ' . $this->lang->line('email_new_password_subject'));
				$this->email->message($message);

				if ($this->email->send())
				{
					$this->set_message('password_change_successful');
					$this->membres_model->trigger_events(array('post_password_change', 'password_change_successful'));
					return TRUE;
				}
				else
				{
					$this->set_error('password_change_unsuccessful');
					$this->membres_model->trigger_events(array('post_password_change', 'password_change_unsuccessful'));
					return FALSE;
				}

			}
		}

		$this->membres_model->trigger_events(array('post_password_change', 'password_change_unsuccessful'));
		return FALSE;
	}

	/**
	 * forgotten_password_check
	 *
	 * @param $code
	 * @author Michael
	 * @return bool
	 */
	public function forgotten_password_check($code)
	{
		$profile = $this->where('forgotten_password_code', $code)->users()->row(); //pass the code to profile

		if (!is_object($profile))
		{
			$this->set_error('password_change_unsuccessful');
			return FALSE;
		}
		else
		{
			if ($this->config->item('forgot_password_expiration', 'el_f2plus_auth') > 0) {
				//Make sure it isn't expired
				$expiration = $this->config->item('forgot_password_expiration', 'el_f2plus_auth');
				if (time() - $profile->forgotten_password_time > $expiration) {
					//it has expired
					$this->clear_forgotten_password_code($code);
					$this->set_error('password_change_unsuccessful');
					return FALSE;
				}
			}
			return $profile;
		}
	}

	/**
	 * register
	 *
	 * @param $identity
	 * @param $password
	 * @param $email
	 * @param array $additional_data
	 * @param array $group_ids
	 * @author Mathew
	 * @return bool
	 */
	public function register($identity, $password, $email, $additional_data = array(), $group_ids) //need to test email activation
	{
		$this->membres_model->trigger_events('pre_account_creation');

		$email_activation = $this->config->item('email_activation', 'el_f2plus_auth');

		$id = $this->membres_model->register($identity, $password, $email, $additional_data, $group_ids);

		if (!$email_activation)
		{
			if ($id !== FALSE)
			{
				$this->set_message('account_creation_successful');
				$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_successful'));
				return $id;
			}
			else
			{
				$this->set_error('account_creation_unsuccessful');
				$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_unsuccessful'));
				return FALSE;
			}
		}
		else
		{
			if (!$id)
			{
				$this->set_error('account_creation_unsuccessful');
				return FALSE;
			}

			// deactivate so the user much follow the activation flow
			$deactivate = $this->membres_model->deactivate($id);

			// the deactivate method call adds a message, here we need to clear that
			$this->membres_model->clear_messages();


			if (!$deactivate)
			{
				$this->set_error('deactivate_unsuccessful');
				$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_unsuccessful'));
				return FALSE;
			}

			$activation_code = $this->membres_model->activation_code;
			$identity        = $this->config->item('identity', 'el_f2plus_auth');
			$user            = $this->membres_model->user($id)->row();

			$data = array(
				'identity'   => $user->{$identity},
				'id'         => $user->id,
				'email'      => $email,
				'activation' => $activation_code,
			);
			if(!$this->config->item('use_ci_email', 'el_f2plus_auth'))
			{
				$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_successful', 'activation_email_successful'));
				$this->set_message('activation_email_successful');
				return $data;
			}
			else
			{
				$message = $this->load->view($this->config->item('email_templates', 'el_f2plus_auth').$this->config->item('email_activate', 'el_f2plus_auth'), $data, true);

				$this->email->clear();
				$this->email->from($this->config->item('admin_email', 'el_f2plus_auth'), $this->config->item('site_title', 'el_f2plus_auth'));
				$this->email->to($email);
				$this->email->subject($this->config->item('site_title', 'el_f2plus_auth') . ' - ' . $this->lang->line('email_activation_subject'));
				$this->email->message($message);

				if ($this->email->send() == TRUE)
				{
					$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_successful', 'activation_email_successful'));
					$this->set_message('activation_email_successful');
					return $id;
				}

			}

			$this->membres_model->trigger_events(array('post_account_creation', 'post_account_creation_unsuccessful', 'activation_email_unsuccessful'));
			$this->set_error('activation_email_unsuccessful');
			return FALSE;
		}
	}

	public function register_sexion($type,$name, $date) //need to test email activation
	{
		$this->membres_model->trigger_events('register_sexion');		

		$id = $this->membres_model->register_sexion($type,$name, $date);

		return $id;
	}

	public function register_book($author, $indexation, $titre, $additional_data = array()) //need to test email activation
	{
		$this->membres_model->trigger_events('register_book');		

		$id = $this->membres_model->register_book($author, $indexation, $titre,$additional_data);

		return $id;
	}

	/**
	 * logout
	 *
	 * @return void
	 * @author Mathew
	 **/
	public function logout()
	{
		$this->membres_model->trigger_events('logout');

		$identity = $this->config->item('identity', 'el_f2plus_auth');

                if (substr(CI_VERSION, 0, 1) == '2')
		{
			$this->session->unset_userdata( array($identity => '', 'id' => '', 'user_id' => '') );
                }
                else
                {
                	$this->session->unset_userdata( array($identity, 'id', 'user_id') );
                }

		// delete the remember me cookies if they exist
		if (get_cookie($this->config->item('identity_cookie_name', 'el_f2plus_auth')))
		{
			delete_cookie($this->config->item('identity_cookie_name', 'el_f2plus_auth'));
		}
		if (get_cookie($this->config->item('remember_cookie_name', 'el_f2plus_auth')))
		{
			delete_cookie($this->config->item('remember_cookie_name', 'el_f2plus_auth'));
		}

		// Destroy the session
		$this->session->sess_destroy();

		//Recreate the session
		if (substr(CI_VERSION, 0, 1) == '2')
		{
			$this->session->sess_create();
		}
		else
		{
			if (version_compare(PHP_VERSION, '7.0.0') >= 0) {
				session_start();
			}
			$this->session->sess_regenerate(TRUE);
		}

		$this->set_message('logout_successful');
		return TRUE;
	}

	/**
	 * logged_in
	 *
	 * @return bool
	 * @author Mathew
	 **/
	public function logged_in()
	{
		$this->membres_model->trigger_events('logged_in');
		$recheck= $this->membres_model->recheck_session();
		//print_r($_COOKIE);
		//die($recheck."  ".get_cookie($this->config->item('identity_cookie_name', 'el_f2plus_auth')));
        //auto-login the user if they are remembered
        if ( ! $recheck && get_cookie($this->config->item('identity_cookie_name', 'el_f2plus_auth')) && get_cookie($this->config->item('remember_cookie_name', 'el_f2plus_auth')))
		{
			$recheck = $this->membres_model->login_remembered_user();
		}
		return $recheck;
	}

	/**
	 * logged_in
	 *
	 * @return integer
	 * @author jrmadsen67
	 **/
	public function get_user_id()
	{
		$user_id = $this->session->userdata('user_id');
		if (!empty($user_id))
		{
			return $user_id;
		}
		return null;
	}


	/**
	 * is_admin
	 *
	 * @return bool
	 * @author Ben Edmunds
	 **/
	public function is_admin($id=false)
	{
		$this->membres_model->trigger_events('is_admin');

		$admin_group = $this->config->item('admin_group', 'el_f2plus_auth');

		return $this->in_group($admin_group, $id);
	}

	public function is_sp($id=false)
	{
		$this->membres_model->trigger_events('is_sp');

		$sp_group = $this->config->item('sp_group', 'el_f2plus_auth');

		return $this->in_group($sp_group, $id);
	}

	public function is_sbp($id=false)
	{
		$this->membres_model->trigger_events('is_sbp');

		$sbp_group = $this->config->item('sbp_group', 'el_f2plus_auth');

		return $this->in_group($sbp_group, $id);
	}

	public function is_sa($id=false)
	{
		$this->membres_model->trigger_events('is_sa');

		$sa_group = $this->config->item('sa_group', 'el_f2plus_auth');

		return $this->in_group($sa_group, $id);
	}

	public function is_members($id=false)
	{
		$this->membres_model->trigger_events('is_members');

		$membre_group = $this->config->item('default_group', 'el_f2plus_auth');

		return $this->in_group($membre_group, $id);
	}

	/**
	 * in_group
	 *
	 * @param mixed group(s) to check
	 * @param bool user id
	 * @param bool check if all groups is present, or any of the groups
	 *
	 * @return bool
	 * @author Phil Sturgeon
	 **/
	public function in_group($check_group, $id=false, $check_all = false)
	{
		$this->membres_model->trigger_events('in_group');

		$id || $id = $this->session->userdata('user_id');

		if (!is_array($check_group))
		{
			$check_group = array($check_group);
		}

		if (isset($this->_cache_user_in_group[$id]))
		{
			$groups_array = $this->_cache_user_in_group[$id];
		}
		else
		{
			$users_groups = $this->membres_model->get_users_groups($id)->result();
			$groups_array = array();
			foreach ($users_groups as $group)
			{
				$groups_array[$group->id] = $group->name;
			}
			$this->_cache_user_in_group[$id] = $groups_array;
		}
		foreach ($check_group as $key => $value)
		{
			$groups = (is_string($value)) ? $groups_array : array_keys($groups_array);

			/**
			 * if !all (default), in_array
			 * if all, !in_array
			 */
			if (in_array($value, $groups) xor $check_all)
			{
				/**
				 * if !all (default), true
				 * if all, false
				 */
				return !$check_all;
			}
		}

		/**
		 * if !all (default), false
		 * if all, true
		 */
		return $check_all;
	}

}
